/**
* \file: EndpointDataSharing.h
*
* \version: 0.4
*
* \release: $Name:$
*
* Transmission mechanism to share data between endpoints
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_ENDPOINT_DATA_SHARING_H
#define USPI_ENDPOINT_DATA_SHARING_H

#include <map>
#include <pthread_adit.h>
#include <wayland-client-protocol.h>
#include <uspi/Singleton.h>

namespace adit { namespace uspi {

/** @interface  SharedDataReceiver
 *  @brief      Receiver class to get data from other endpoints over SharedDataSender
 */
class SharedDataReceiver
{
public:
    SharedDataReceiver(void* inSessionCtx) : mSessionCtx(inSessionCtx) { }
    virtual ~SharedDataReceiver() { mSessionCtx = nullptr; }

    /** @brief  callback initiated when the wl_display provider is active
     *  @param  inDisplay: wl_display being received when the provider is active
     */
    virtual void onSurfaceDataProvided(wl_display* inDisplay) {
        (void)inDisplay;
    }
    /** @brief  callback initiated when the wl_display provider is inactive
     */
    virtual void onSurfaceDataExpired() { }

    /** @brief  callback initiated when video resolution data is provided
     *  @param  inWidth: width of the surface
     *  @param  inHeight: height of the surface
     *  @param  inMarginX: x-coordinate of margin to be left from the surface
     *  @param  inMarginY: y-coordinate of margin to be left from the surface
     */
    virtual void onResolutionDataProvided(int inWidth, int inHeight, int inMarginX, int inMarginY)
    {
        (void)inWidth;
        (void)inHeight;
        (void)inMarginX;
        (void)inMarginY;
    }

    void* sessionCtx() { return mSessionCtx; }
private:
    void* mSessionCtx;
};

/** @class  SharedDataSender
 *  @brief  Singleton class to transfer data to other endpoints
 */
class SharedDataSender : public Singleton<SharedDataSender>
{
public:
    SharedDataSender();
    virtual ~SharedDataSender();

    /** @brief  subscribe to the sender to get notified when there is information available
     *  @param  inReceiver: the receiver to be subscribed
     *  @return 0 on successful subscription,
     *          -1 if already subscribed,
     *          error codes > 0 are defined in errno.h.
     */
    int subscribeForDataSharing(SharedDataReceiver& inReceiver);
    /** @brief  unsubscribe from the sender
     *  @param  inReceiver: the receiver to be unsubscribed
     *  @return 0 on successful unsubscription,
     *          -1 if already unsubscribed,
     *          error codes > 0 are defined in errno.h.
     */
    int unsubscribeFromDataSharing(SharedDataReceiver& inReceiver);

    /** @brief  transmit surface data (reference to wayland surface) to the subscribed receivers within the same session
     *  @param  inSessionCtx: the session identifier
     *  @param  inDisplay: wayland display to be transferred to receiver
     *  @return 0 on successful transmission,
     *          -1 if no receiver with the same SessionCtx is registered,
     *          error codes > 0 are defined in errno.h.
     */
    int transmitSurfaceData(void* inSessionCtx, wl_display* inDisplay);
    /** @brief  notify the subscribed receivers within the same session, that the surface data is invalid now
     *  @param  inSessionCtx: the session identifier
     *  @return 0 on successful notification,
     *          -1 if no receiver with the same SessionCtx is registered,
     *          error codes > 0 are defined in errno.h.
     */
    int notifySurfaceDataExpiration(void* inSessionCtx);

    /** @brief  transmit resolution data to the subscribed receivers within the same session
     *  @param  inSessionCtx: the session identifier
     *  @param  inWidth: width of the surface
     *  @param  inHeight: height of the surface
     *  @param  inMarginX: x-coordinate of margin to be left from the surface [default value = 0]
     *  @param  inMarginY: y-coordinate of margin to be left from the surface [default value = 0]
     *  @return 0 on successful transmission,
     *          -1 if no receiver with the same SessionCtx is registered,
     *          error codes > 0 are defined in errno.h.
     */
    int transmitResolutionData(void* inSessionCtx, int inWidth, int inHeight, int inMarginX = 0, int inMarginY = 0);

private:
    /* SharedDataReceiver / sessionCtx pairs subscribed to get surface data */
    std::map<void*, SharedDataReceiver*> mReceiversList;
    pthread_mutex_t mReceiversListMutex;

    SharedDataReceiver* findReceiverBySessionCtx(void* inSessionCtx);
};

} } /* namespace adit { namespace uspi { */

#endif /* USPI_ENDPOINT_DATA_SHARING_H */
